# 機能設計書 53-音声操作（Audio Operations）

## 概要

本ドキュメントは、TensorFlowにおける音声操作機能の設計を記述する。WAVファイルのデコード・エンコード、オーディオスペクトログラム計算、MFCC（メル周波数ケプストラム係数）抽出のオペレーションを提供する。

### 本機能の処理概要

**業務上の目的・背景**：音声認識・音声コマンド検出・音響イベント検出などの機械学習タスクにおいて、音声データの前処理（WAVデコード、スペクトログラム変換、MFCC特徴量抽出）は不可欠である。本機能はこれらの音声信号処理パイプラインを構成するためのC++オペレーション群を提供する。

**機能の利用シーン**：マイクからの音声入力の前処理（PCMデータのWAVデコード）、音声認識モデルへの入力特徴量生成（スペクトログラム計算、MFCC抽出）、音声ファイルの保存（WAVエンコード）で利用される。

**主要な処理内容**：
1. DecodeWav: WAV形式音声ファイルのデコード（バイト列→浮動小数点テンソル＋サンプルレート）
2. EncodeWav: 浮動小数点テンソル＋サンプルレートからWAV形式へのエンコード
3. AudioSpectrogram: 短時間フーリエ変換（STFT）ベースのスペクトログラム計算
4. Mfcc: スペクトログラムからメル周波数ケプストラム係数を抽出

**関連システム・外部連携**：Android AudioRecord API（マイク入力）との連携。音声認識モデル（conv_actions_frozen.pb等）への入力供給。

**権限による制御**：特段の権限制御はない。マイクアクセスはOS側の権限管理に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | 音声認識画面 | 主機能 | マイクからの音声入力をAudioRecordで取得しPCM16ビットデータとして録音 |

## 機能種別

計算処理 / 信号処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| contents | string | Yes | WAVファイルのバイト列（DecodeWav用） | rank=0 スカラー |
| audio | float | Yes | 音声波形テンソル [samples, channels]（EncodeWav用） | rank=2 |
| sample_rate | int32 | Yes | サンプリングレート（EncodeWav用） | rank=0 スカラー |
| input | float | Yes | 音声波形テンソル [samples, channels]（AudioSpectrogram用） | rank=2 |
| spectrogram | float | Yes | スペクトログラムテンソル [channels, time, frequency]（Mfcc用） | rank=3 |
| window_size | int（属性） | Yes | スペクトログラムの窓サイズ | > 1 |
| stride | int（属性） | Yes | スペクトログラムのストライド | > 0 |
| desired_channels | int（属性） | No | デコード時の希望チャネル数（-1=自動） | -1以上 |
| desired_samples | int（属性） | No | デコード時の希望サンプル数（-1=自動） | -1以上 |
| dct_coefficient_count | int（属性） | No | MFCC係数の数（デフォルト13） | 正整数 |

### 入力データソース

マイク入力（AudioRecord）、ファイルシステム上のWAVファイル、tf.data Datasetからの音声テンソル。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| audio | float | デコードされた音声テンソル [samples, channels]（DecodeWav） |
| sample_rate | int32 | サンプリングレート（DecodeWav） |
| contents | string | エンコードされたWAVバイト列（EncodeWav） |
| spectrogram | float | スペクトログラム [channels, time_steps, frequency_bins]（AudioSpectrogram） |
| output | float | MFCC特徴量 [channels, time_steps, dct_coefficients]（Mfcc） |

### 出力先

後続の計算グラフ（音声認識モデル入力、ファイル保存）。

## 処理フロー

### 処理シーケンス

```
1. WAVファイルデコード
   └─ DecodeWav: バイト列 → float音声テンソル [samples, channels] + サンプルレート
2. スペクトログラム計算
   └─ AudioSpectrogram: 音声波形 → STFT → [channels, time_steps, frequency_bins]
3. MFCC特徴量抽出
   └─ Mfcc: スペクトログラム → メルフィルタバンク → DCT → [channels, time_steps, coefficients]
4. WAVファイルエンコード（必要時）
   └─ EncodeWav: float音声テンソル + サンプルレート → WAVバイト列
```

### フローチャート

```mermaid
flowchart TD
    A[音声入力: WAVバイト列] --> B[DecodeWav]
    B --> C[音声テンソル + サンプルレート]
    C --> D[AudioSpectrogram]
    D --> E[スペクトログラム]
    E --> F[Mfcc]
    F --> G[MFCC特徴量]
    G --> H[音声認識モデル入力]
    C --> I[EncodeWav]
    I --> J[WAVバイト列出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-53-01 | スペクトログラム窓サイズ | window_sizeは1より大きくなければならない | AudioSpectrogram |
| BR-53-02 | ストライド正値 | strideは0より大きくなければならない | AudioSpectrogram |
| BR-53-03 | 周波数ビン数 | 出力周波数ビン数 = 1 + NextPowerOfTwo(window_size) / 2 | AudioSpectrogram |
| BR-53-04 | 時間ステップ数 | output_length = max(0, 1 + (input_length - window_size) / stride) | AudioSpectrogram |
| BR-53-05 | MFCC出力チャネル | MFCC出力の最後の次元はdct_coefficient_count（デフォルト13） | Mfcc |

### 計算ロジック

- **スペクトログラム**: 入力波形に対してwindow_sizeの窓でstrideずつスライドしながらFFTを適用。magnitude_squared=trueの場合はパワースペクトル。
- **MFCC**: スペクトログラムにメルフィルタバンク（filterbank_channel_count=40、lower=20Hz、upper=4000Hz）を適用後、DCTでdct_coefficient_count個の係数を抽出。

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | 窓サイズエラー | window_size <= 1 | 2以上の値を指定 |
| InvalidArgument | ストライドエラー | stride <= 0 | 正の値を指定 |
| InvalidArgument | チャネル数エラー | desired_channels < -1 | -1以上の値を指定 |
| InvalidArgument | サンプル数エラー | desired_samples < -1 | -1以上の値を指定 |

### リトライ仕様

音声処理はステートレスであり、リトライに特別な考慮は不要。

## トランザクション仕様

トランザクション制御は行わない。

## パフォーマンス要件

リアルタイム音声認識では低レイテンシ（数十ミリ秒単位）での処理が求められる場合がある。

## セキュリティ考慮事項

マイク入力データのプライバシー保護に注意が必要。

## 備考

audio_ops.ccは約168行のコンパクトなファイルで、4つのオペレーション（DecodeWav、EncodeWav、AudioSpectrogram、Mfcc）を定義している。スペクトログラムの周波数ビン数計算にはNextPowerOfTwo関数（bits.hから）を使用している。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

音声テンソルの形状表現を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | 30-61行目: DecodeWavShapeFn - 出力形状 [samples, channels] の推論 |
| 1-2 | audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | 71-111行目: SpectrogramShapeFn - 出力形状 [channels, time, freq] の推論 |
| 1-3 | audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | 113-131行目: MfccShapeFn - 出力形状 [channels, time, dct_coefficients] の推論 |

**読解のコツ**: 音声テンソルの軸順序はTensorFlow特有で、[samples, channels]（波形）と[channels, time, frequency]（スペクトログラム）である。画像テンソル（NHWC）とは異なる配列順序に注意。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | 135-141行目: DecodeWav - WAVデコード |
| 2-2 | audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | 143-147行目: EncodeWav - WAVエンコード |
| 2-3 | audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | 149-155行目: AudioSpectrogram - STFT |
| 2-4 | audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | 157-165行目: Mfcc - MFCC抽出 |

**主要処理フロー**:
1. **135-141行目**: DecodeWav: string入力、desired_channels/desired_samples属性、audio(float)+sample_rate(int32)出力
2. **149-155行目**: AudioSpectrogram: float入力、window_size/stride/magnitude_squared属性、spectrogram(float)出力
3. **157-165行目**: Mfcc: spectrogram+sample_rate入力、upper/lower_frequency_limit/filterbank_channel_count/dct_coefficient_count属性

### プログラム呼び出し階層図

```
audio_ops.cc (Op定義)
    |
    +-- ヘルパー関数
    |    +-- DecodeWavShapeFn (30-61行目)
    |    +-- EncodeWavShapeFn (63-69行目)
    |    +-- SpectrogramShapeFn (71-111行目)
    |    +-- MfccShapeFn (113-131行目)
    |
    +-- Op登録
         +-- DecodeWav (135行目)
         +-- EncodeWav (143行目)
         +-- AudioSpectrogram (149行目)
         +-- Mfcc (157行目)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

WAVバイト列 --------> DecodeWav -----------------> audio [samples, channels]
                                                    + sample_rate (int32)
                                                         |
audio [samples, ch] --> AudioSpectrogram ----------> spectrogram [ch, time, freq]
                        (window_size, stride)             |
                                                         v
spectrogram ---------> Mfcc ----------------------> output [ch, time, dct_coeff]
+ sample_rate          (filterbank, freq limits)

audio + sample_rate -> EncodeWav -----------------> WAVバイト列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| audio_ops.cc | `tensorflow/core/ops/audio_ops.cc` | ソース | 音声オペレーション定義（4 Op） |
| bits.h | `tensorflow/core/lib/core/bits.h` | ヘッダ | NextPowerOfTwo関数（FFTサイズ計算用） |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | 共通形状推論関数 |
| op.h | `tensorflow/core/framework/op.h` | ヘッダ | REGISTER_OPマクロ定義 |
